// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef COTA_ARRAY_HPP
#define COTA_ARRAY_HPP

#include <cota/detail/array_base.hpp>
#include <memory>

namespace cota {
template <typename T, typename Allocator = std::allocator<T>>
class array : public detail::array_base {
private:
  using passed_allocator = Allocator;

public:
  using value_type = T;
  using allocator_type =
    typename std::allocator_traits<passed_allocator>::template rebind_alloc<value_type>;

private:
  using alloc_traits = std::allocator_traits<allocator_type>;

public:
  using size_type = std::size_t;
  using difference_type = std::ptrdiff_t;
  using reference = value_type&;
  using const_reference = const value_type&;
  using pointer = typename alloc_traits::pointer;
  using const_pointer = typename alloc_traits::const_pointer;

  using iterator = pointer;
  using const_iterator = const_pointer;
  using reverse_iterator = std::reverse_iterator<iterator>;
  using const_reverse_iterator = std::reverse_iterator<const_iterator>;

public:
  array() = default;
  explicit array(allocator_type const& alloc) noexcept;
  explicit array(size_type count);
  array(size_type count, allocator_type const& alloc);

  array(array&& other) noexcept;
  array(array&& other, allocator_type const& alloc) noexcept;

  auto operator=(array&& other) noexcept -> array&;

  ~array();

private:
  array(array const& other);
  array(array const& other, allocator_type const& alloc);
  auto operator=(array const& other) -> array&;

public:
  auto clone() const -> array;

  template <typename Alloc>
  auto clone(Alloc const& alloc) const -> array;

public:
  auto operator[](size_type index) -> reference;
  auto operator[](size_type index) const -> const_reference;

  auto begin() noexcept -> iterator;
  auto begin() const noexcept -> const_iterator;

  auto end() noexcept -> iterator;
  auto end() const noexcept -> const_iterator;

  auto size() const noexcept -> size_type;

  auto data() noexcept -> pointer;
  auto data() const noexcept -> const_pointer;

private:
  allocator_type alloc_;
  size_type capacity_{};
  pointer begin_;
  pointer end_;
};

// Deduction Guides
template <typename T, typename Alloc>
array(array<T, Alloc>&&) -> array<T, Alloc>;

template <typename T, typename Allocator>
array<T, Allocator>::array(allocator_type const& alloc) noexcept
: alloc_(alloc), begin_(nullptr), end_(nullptr)
{
}

template <typename T, typename Allocator>
array<T, Allocator>::array(size_type count, allocator_type const& alloc)
: alloc_(alloc), capacity_(count), begin_(alloc_.allocate(capacity_)), end_(begin_ + count)
{
}

// NOLINTNEXTLINE(cppcoreguidelines-pro-type-member-init)
template <typename T, typename Allocator>
array<T, Allocator>::array(size_type count) : array(count, allocator_type())
{
}

template <typename T, typename Allocator>
array<T, Allocator>::array(array&& other) noexcept
: alloc_(std::move(other.alloc_)),
  capacity_(other.capacity_),
  begin_(other.begin_),
  end_(other.end_)
{
  other.begin_ = nullptr;
  other.end_ = nullptr;
  other.capacity_ = 0;
}

template <typename T, typename Allocator>
array<T, Allocator>::array(array&& other, allocator_type const& alloc) noexcept : alloc_(alloc)
{
  if (alloc_ == other.alloc_) {
    capacity_ = other.capacity_;
    begin_ = other.begin_;
    end_ = other.end_;

    other.begin_ = nullptr;
    other.end_ = nullptr;
    other.capacity_ = 0;
  }
  else {
    capacity_ = other.size();
    begin_ = alloc_.allocate(capacity_);
    end_ = begin_ + capacity_;
    std::uninitialized_move(other.begin_, other.end_, begin_);
  }
}

template <typename T, typename Allocator>
array<T, Allocator>::array(array const& other)
: array(other, alloc_traits::select_on_container_copy_construction(other.alloc_))
{
}

template <typename T, typename Allocator>
array<T, Allocator>::array(array const& other, allocator_type const& alloc)
: alloc_(alloc),
  capacity_(other.size()),
  begin_(alloc_.allocate(capacity_)),
  end_(begin_ + capacity_)
{
  std::uninitialized_copy(other.begin_, other.end_, begin_);
}

template <typename T, typename Allocator>
auto array<T, Allocator>::operator=(array&& other) noexcept -> array&
{
  if (this != &other) {
    if (alloc_ == other.alloc_) {
      alloc_.deallocate(begin_, capacity_);
      capacity_ = other.capacity_;
      begin_ = other.begin_;
      end_ = other.end_;

      other.begin_ = nullptr;
      other.end_ = nullptr;
      other.capacity_ = 0;
    }
    else {
      if (alloc_traits::propagate_on_container_move_assignment::value) {
        alloc_ = std::move(other.alloc_);
      }
      alloc_.deallocate(begin_, capacity_);
      capacity_ = other.size();
      alloc_.allocate(begin_, capacity_);
      std::uninitialized_copy(other.begin_, other.end_, begin_);
      other.alloc_.deallocate(other.begin_, other.capacity_);
      other.begin_ = nullptr;
      other.end_ = nullptr;
      other.capacity_ = 0;
    }
  }
  return *this;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::operator=(array const& other) -> array&
{
  if (this != &other) {
    if (alloc_ == other.alloc_) {
    }
    else {
      if (alloc_traits::propagate_on_container_copy_assignment::value) {
        alloc_ = std::move(other.alloc_);
      }
      alloc_.deallocate(begin_, capacity_);
      capacity_ = other.size();
      alloc_.allocate(begin_, capacity_);
      std::uninitialized_copy(other.begin_, other.end_, begin_);
    }
  }
  return *this;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::clone() const -> array
{
  return clone(alloc_);
}

template <typename T, typename Allocator>
template <typename Alloc>
auto array<T, Allocator>::clone(Alloc const& alloc) const -> array
{
  array result(size(), alloc);
  std::uninitialized_copy(begin_, end_, result.begin_);
  return result;
}

template <typename T, typename Allocator>
array<T, Allocator>::~array()
{
  alloc_.deallocate(begin_, capacity_);
  capacity_ = 0;
  begin_ = nullptr;
  end_ = nullptr;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::operator[](size_type index) -> reference
{
  return begin_[index];
}

template <typename T, typename Allocator>
auto array<T, Allocator>::operator[](size_type index) const -> const_reference
{
  return begin_[index];
}

template <typename T, typename Allocator>
auto array<T, Allocator>::begin() noexcept -> iterator
{
  return begin_;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::begin() const noexcept -> const_iterator
{
  return begin_;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::end() noexcept -> iterator
{
  return end_;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::end() const noexcept -> const_iterator
{
  return end_;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::size() const noexcept -> size_type
{
  return static_cast<size_type>(end_ - begin_);
}

template <typename T, typename Allocator>
auto array<T, Allocator>::data() noexcept -> pointer
{
  return begin_;
}

template <typename T, typename Allocator>
auto array<T, Allocator>::data() const noexcept -> const_pointer
{
  return begin_;
}
} // namespace cota

#endif // COTA_ARRAY_HPP
