// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef NOLA_STATIC_ARRAY_HPP
#define NOLA_STATIC_ARRAY_HPP

#include <nola/qualifier.hpp>
#include <nola/wrap_iterator.hpp>

namespace nola {
template <typename T, std::size_t N>
class static_array {

public:
  using value_type = T;
  using reference = value_type&;
  using const_reference = value_type const&;

  using pointer = value_type*;
  using const_pointer = value_type const*;

  using iterator = wrap_iterator<pointer>;
  using const_iterator = wrap_iterator<const_pointer>;

  using size_type = std::size_t;
  using difference_type = std::ptrdiff_t;

  using reverse_iterator = std::reverse_iterator<iterator>;
  using const_reverse_iterator = std::reverse_iterator<const_iterator>;

public:
  static_array() = default;

  template <typename... U>
  NOLA_HOST_DEVICE
  constexpr explicit static_array(U... u)
    requires(sizeof...(U) == N and std::conjunction_v<std::is_same<T, U>...>);

public:
  NOLA_HOST_DEVICE
  constexpr auto begin() const noexcept -> const_iterator;

  NOLA_HOST_DEVICE
  constexpr auto end() const noexcept -> const_iterator;

  NOLA_HOST_DEVICE
  constexpr auto begin() noexcept -> iterator;

  NOLA_HOST_DEVICE
  constexpr auto end() noexcept -> iterator;

  NOLA_HOST_DEVICE
  constexpr auto operator[](size_type index) const noexcept -> const_reference;

  NOLA_HOST_DEVICE
  constexpr auto operator[](size_type index) noexcept -> reference;

  NOLA_HOST_DEVICE
  constexpr auto operator=(value_type const& v) noexcept -> static_array&;

  NOLA_HOST_DEVICE
  constexpr auto size() const noexcept -> size_type;

private:
  // NOLINTNEXTLINE(modernize-avoid-c-arrays)
  value_type value_[N];
};

template <class T, class... U>
static_array(T, U...) -> static_array<T, sizeof...(U)>;

template <typename T, typename... Args>
constexpr auto to_static_array(Args&&... args) -> nola::static_array<T, sizeof...(Args)>
{
  return {{std::forward<Args>(args)...}};
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto operator==(static_array<T, N> const& a, static_array<T, N> const& b) -> bool
{
  for (std::size_t i = 0; i < N; ++i) {
    if (a[i] != b[i]) {
      return false;
    }
  }
  return true;
}

// -------------------------------------------------------------------------------------------------
// Implementation
// -------------------------------------------------------------------------------------------------

template <typename T, std::size_t N>
template <typename... U>
NOLA_HOST_DEVICE
constexpr static_array<T, N>::static_array(U... u)
  requires(sizeof...(U) == N and std::conjunction_v<std::is_same<T, U>...>)
: value_{u...}
{
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::begin() const noexcept -> const_iterator
{
  return wrap_iterator<const_pointer>(&value_[0]);
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::end() const noexcept -> const_iterator
{
  return wrap_iterator<const_pointer>(&value_[0] + N);
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::begin() noexcept -> iterator
{
  return wrap_iterator<pointer>(&value_[0]);
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::end() noexcept -> iterator
{
  return wrap_iterator<pointer>(&value_[0] + N);
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::operator[](size_type index) const noexcept -> const_reference
{
  return value_[index];
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::operator[](size_type index) noexcept -> reference
{
  return value_[index];
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::operator=(value_type const& v) noexcept -> static_array&
{
  for (std::size_t i = 0; i < N; ++i) {
    value_[i] = v;
  }
  return *this;
}

template <typename T, std::size_t N>
NOLA_HOST_DEVICE
constexpr auto static_array<T, N>::size() const noexcept -> size_type
{
  return N;
}

} // namespace nola

#endif // NOLA_STATIC_ARRAY_HPP
