// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_CELL_ARRAY_HPP
#define ZELL_IO_VTK_UGRID_CELL_ARRAY_HPP

#include <bati/smath.hpp>
#include <bati/vector.hpp>
#include <memory>
#include <nola/wrap_iterator.hpp>
#include <vector>
#include <zell/io/vtk/ugrid/cell_type.hpp>
#include <zell/io/vtk/ugrid/cell_view.hpp>

namespace zell::io::vtk::ugrid {
template <typename Alloc = std::allocator<std::byte>>
class cell_array {
public:
  using value_type = cell_view;
  using allocator_type = Alloc;
  using reference = value_type&;
  using const_reference = value_type const&;

private:
  using alloc_traits = std::allocator_traits<allocator_type>;

  using value_allocator = typename alloc_traits::template rebind_alloc<cell_view>;
  using index_allocator = typename alloc_traits::template rebind_alloc<std::size_t>;
  using type_allocator = typename alloc_traits::template rebind_alloc<cell_type>;

public:
  using cell_array_type = std::vector<value_type, value_allocator>;
  using index_array_type = std::vector<std::size_t, index_allocator>;
  using cell_type_array_type = std::vector<cell_type, type_allocator>;

public:
  using pointer = typename cell_array_type::pointer;
  using const_pointer = typename cell_array_type::const_pointer;

  using iterator = typename nola::wrap_iterator<pointer>;
  using const_iterator = typename nola::wrap_iterator<const_pointer>;

  using size_type = std::size_t;
  using difference_type = std::ptrdiff_t;

  using reverse_iterator = std::reverse_iterator<iterator>;
  using const_reverse_iterator = std::reverse_iterator<const_iterator>;

public:
  cell_array() = default;
  cell_array(cell_array&& g) noexcept = default;
  auto operator=(cell_array&& g) noexcept -> cell_array& = default;
  ~cell_array() = default;

  explicit cell_array(allocator_type const& a);
  cell_array(cell_array&& g, allocator_type a) noexcept;

private:
  cell_array(cell_array const& g) = default;
  auto operator=(cell_array const& g) -> cell_array& = default;

  cell_array(cell_array const& g, allocator_type a);

public:
  auto clone() const -> cell_array;

  template <typename OtherAlloc>
  auto clone(OtherAlloc const& other) const -> cell_array<OtherAlloc>;

public:
  static auto create(
    index_array_type index_array, cell_type_array_type cell_type_array, allocator_type const& a)
    -> cell_array;

public:
  auto size() const noexcept -> size_type;
  auto get_allocator() const noexcept -> allocator_type;

public:
  auto begin() const noexcept -> const_iterator;
  auto begin() noexcept -> iterator;

  auto end() const noexcept -> const_iterator;
  auto end() noexcept -> iterator;

  auto operator[](size_type index) noexcept -> reference;
  auto operator[](size_type index) const noexcept -> const_reference;

private:
  allocator_type allocator_;
  cell_array_type array_;
  cell_type_array_type cell_type_array_;
  index_array_type index_array_;
};

template <typename Alloc>
cell_array<Alloc>::cell_array(allocator_type const& a) : array_(a)
{
}

template <typename Alloc>
cell_array<Alloc>::cell_array(cell_array&& g, allocator_type a) noexcept
: allocator_(a),
  array_(std::move(g.array_), a),
  cell_type_array_(std::move(g.cell_type_array_), a),
  index_array_(std::move(g.index_array_), a)
{
}

template <typename Alloc>
cell_array<Alloc>::cell_array(cell_array const& g, allocator_type a)
: allocator_(a),
  array_(g.array_, a),
  cell_type_array_(g.cell_type_array_, a),
  index_array_(g.index_array_, a)
{
}

template <typename Alloc>
auto cell_array<Alloc>::create(
  index_array_type index_array, cell_type_array_type cell_type_array, allocator_type const& a)
  -> cell_array
{
  cell_array ca;
  ca.allocator_ = a;
  ca.index_array_ = std::move(index_array);
  ca.cell_type_array_ = std::move(cell_type_array);
  ca.array_.resize(ca.cell_type_array_.size());

  std::size_t index = 0;
  for (std::size_t i = 0; i < ca.cell_type_array_.size(); ++i) {
    auto nvertex = cell_nvertex(ca.cell_type_array_[i]);
    ca.array_[i] = cell_view(
      ca.cell_type_array_[i], &ca.index_array_[index], &ca.index_array_[index + nvertex]);

    index += nvertex;
  }

  return ca;
}

template <typename Alloc>
auto cell_array<Alloc>::clone() const -> cell_array
{
  return cell_array(*this);
}

template <typename Alloc>
template <typename OtherAlloc>
auto cell_array<Alloc>::clone(OtherAlloc const& other) const -> cell_array<OtherAlloc>
{
  return cell_array(*this, other);
}

template <typename Alloc>
auto cell_array<Alloc>::size() const noexcept -> size_type
{
  return array_.size();
}

template <typename Alloc>
auto cell_array<Alloc>::get_allocator() const noexcept -> allocator_type
{
  return allocator_;
}

template <typename Alloc>
auto cell_array<Alloc>::begin() const noexcept -> const_iterator
{
  return array_.begin();
}

template <typename Alloc>
auto cell_array<Alloc>::begin() noexcept -> iterator
{
  return array_.begin();
}

template <typename Alloc>
auto cell_array<Alloc>::end() const noexcept -> const_iterator
{
  return array_.end();
}
template <typename Alloc>
auto cell_array<Alloc>::end() noexcept -> iterator
{
  return array_.end();
}

template <typename Alloc>
auto cell_array<Alloc>::operator[](size_type index) noexcept -> reference
{
  return array_[index];
}

template <typename Alloc>
auto cell_array<Alloc>::operator[](size_type index) const noexcept -> const_reference
{
  return array_[index];
}

} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_CELL_ARRAY_HPP
