// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_DATASET_HPP
#define ZELL_IO_VTK_UGRID_DATASET_HPP

#include <memory>
#include <nola/array.hpp>
#include <nola/owning_ptr.hpp>
#include <utility>
#include <vector>
#include <zell/io/vtk/ugrid/error.hpp>
#include <zell/io/vtk/ugrid/pmdata.hpp>
#include <zell/io/vtk/ugrid/scalar_data.hpp>
#include <zell/io/vtk/ugrid/vector_data.hpp>

namespace zell::io::vtk::ugrid {
template <typename Alloc = std::allocator<std::byte>>
class dataset {

public:
  using value_type = nola::owning_ptr<pmdata, Alloc>;

  using allocator_type = typename std::allocator_traits<Alloc>::template rebind_alloc<value_type>;

  using storage_type = std::vector<value_type, allocator_type>;
  using size_type = typename storage_type::size_type;
  using reference = typename storage_type::reference;
  using const_reference = typename storage_type::const_reference;
  using pointer = typename storage_type::pointer;
  using const_pointer = typename storage_type::const_pointer;
  using iterator = typename storage_type::iterator;
  using const_iterator = typename storage_type::const_iterator;

public:
  dataset() = default;
  explicit dataset(allocator_type const& a);
  dataset(size_type n, allocator_type const& a);
  explicit dataset(size_type n);

public:
  auto operator[](size_type i) noexcept -> reference;
  auto operator[](size_type i) const noexcept -> const_reference;

  auto begin() noexcept -> iterator;
  auto begin() const noexcept -> const_iterator;

  auto end() noexcept -> iterator;
  auto end() const noexcept -> const_iterator;

public:
  auto add_data(value_type data) -> error { storage_.push_back(std::move(data)); }

private:
  allocator_type alloc_;
  storage_type storage_;
};

template <typename Alloc>
dataset<Alloc>::dataset(allocator_type const& a) : alloc_(a), storage_(a)
{
}

template <typename Alloc>
dataset<Alloc>::dataset(size_type const n, allocator_type const& a) : alloc_(a), storage_(n, alloc_)
{
}

template <typename Alloc>
dataset<Alloc>::dataset(size_type const n) : storage_(n)
{
}

template <typename Alloc>
auto dataset<Alloc>::operator[](size_type const i) noexcept -> reference
{
  return storage_[i];
}

template <typename Alloc>
auto dataset<Alloc>::operator[](size_type const i) const noexcept -> const_reference
{
  return storage_[i];
}

template <typename Alloc>
auto dataset<Alloc>::begin() noexcept -> iterator
{
  return storage_.begin();
}

template <typename Alloc>
auto dataset<Alloc>::begin() const noexcept -> const_iterator
{
  return storage_.begin();
}

template <typename Alloc>
auto dataset<Alloc>::end() noexcept -> iterator
{
  return storage_.end();
}

template <typename Alloc>
auto dataset<Alloc>::end() const noexcept -> const_iterator
{
  return storage_.end();
}
} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_DATASET_HPP
