// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_DETAIL_IO_HPP
#define ZELL_IO_VTK_UGRID_DETAIL_IO_HPP

#include <fstream>
#include <jnum/traits.hpp>
#include <memory>
#include <nlohmann/json.hpp>
#include <nola/owning_ptr.hpp>
#include <tl/expected.hpp>
#include <vector>
#include <zell/io/vtk/ugrid/cell_array.hpp>
#include <zell/io/vtk/ugrid/cell_nvertex.hpp>
#include <zell/io/vtk/ugrid/cell_type.hpp>
#include <zell/io/vtk/ugrid/dataset.hpp>
#include <zell/io/vtk/ugrid/detail/float_string.hpp>
#include <zell/io/vtk/ugrid/detail/metadata.hpp>
#include <zell/io/vtk/ugrid/detail/scalar_header.hpp>
#include <zell/io/vtk/ugrid/detail/vector_header.hpp>
#include <zell/io/vtk/ugrid/error.hpp>
#include <zell/io/vtk/ugrid/pmdata.hpp>
#include <zell/io/vtk/ugrid/point_array.hpp>
#include <zell/io/vtk/ugrid/scalar_data.hpp>
#include <zell/io/vtk/ugrid/type_map.hpp>
#include <zell/io/vtk/ugrid/vector_data.hpp>

#include <iostream>

namespace zell::io::vtk::ugrid::detail {
template <typename Alloc>
auto parse_header(std::ifstream& input, Alloc a) -> tl::expected<metadata<Alloc>, error>;

template <typename F, typename Alloc>
auto parse_points(std::ifstream& input, Alloc a) -> tl::expected<point_array<F, Alloc>, error>;

template <typename Alloc>
auto parse_cells(std::ifstream& input, Alloc a) -> tl::expected<cell_array<Alloc>, error>;

template <typename Alloc>
auto parse_scalar_data(std::ifstream& input, Alloc a)
  -> tl::expected<nola::owning_ptr<pmdata, Alloc>, error>;

template <typename Alloc>
auto parse_vector_data(std::ifstream& input, Alloc a)
  -> tl::expected<nola::owning_ptr<pmdata, Alloc>, error>;

template <typename F, typename Alloc>
auto parse_scalar_data_impl(
  std::ifstream& input, std::size_t num_points, scalar_header<Alloc> sh, Alloc a)
  -> tl::expected<scalar_data<F, Alloc>, error>;

template <typename F, typename Alloc>
auto parse_vector_data_impl(std::ifstream& input, std::size_t num_points, Alloc a)
  -> tl::expected<vector_data<F, Alloc>, error>;

template <typename Alloc>
auto parse_scalar_header(std::ifstream& input, Alloc a)
  -> tl::expected<scalar_header<Alloc>, error>;

// -------------------------------------------------------------------------------------------------
// Implementation
// -------------------------------------------------------------------------------------------------

template <typename Alloc>
auto parse_header(std::ifstream& input, Alloc a) -> tl::expected<metadata<Alloc>, error>
{
  std::string version_string(a);
  std::getline(input, version_string);

  auto const exact_header_string = std::string("# vtk DataFile Version 3.0", a);
  if (version_string != exact_header_string) {
    return tl::make_unexpected(error::ill_formed_header);
  }

  std::string zell_id(a);
  input >> zell_id;

  if (zell_id != "zell") {
    return tl::make_unexpected(error::ill_formed_header);
  }

  std::string meta_title(a);
  input >> meta_title;

  std::string meta_header(a);
  input >> meta_header;

  if (meta_header != "meta") {
    return tl::make_unexpected(error::ill_formed_header);
  }

  std::string metadata_string(a);
  input >> metadata_string;

  metadata_string = std::string(metadata_string, a);
  metadata_string.erase(
    std::remove(metadata_string.begin(), metadata_string.end(), '\"'), metadata_string.end());

  auto metadata_json = nlohmann::json::parse(metadata_string);

  std::string data_format;
  input >> data_format;
  if (data_format != "ASCII") {
    return tl::make_unexpected(error::no_support_for_binary);
  }

  auto meta = metadata<Alloc>(
    meta_title,
    static_cast<std::size_t>(metadata_json[0]),
    static_cast<std::size_t>(metadata_json[1]),
    static_cast<std::size_t>(metadata_json[2]),
    static_cast<std::size_t>(metadata_json[3]),
    a);

  return meta;
}

template <typename F, typename Alloc>
auto parse_points(std::ifstream& input, Alloc a) -> tl::expected<point_array<F, Alloc>, error>
{
  using string_type = std::basic_string<char, std::char_traits<char>, Alloc>;
  string_type points_string(a);
  input >> points_string;

  if (points_string != "POINTS") {
    return tl::make_unexpected(error::expected_points);
  }

  std::size_t num_points = 0;
  input >> num_points;

  string_type float_type(a);
  input >> float_type;
  if (float_type != float_string<F>::value) {
    return tl::make_unexpected(error::no_support_for_binary);
  }

  auto points = point_array<F, Alloc>(num_points, a);

  for (std::size_t i = 0; i < num_points; ++i) {
    input >> points[i][0];
    input >> points[i][1];
    input >> points[i][2];
  }
  return points;
}

template <typename Alloc>
auto parse_cells(std::ifstream& input, Alloc a) -> tl::expected<cell_array<Alloc>, error>
{
  using char_allocator_type = typename std::allocator_traits<Alloc>::template rebind_alloc<char>;
  using string_type = std::basic_string<char, std::char_traits<char>, char_allocator_type>;
  string_type cells_string(a);
  input >> cells_string;

  if (cells_string != "CELLS") {
    return tl::make_unexpected(error::expected_cells);
  }

  std::size_t num_cells = 0;
  input >> num_cells;
  std::size_t num_cell_indexes = 0;
  input >> num_cell_indexes;

  using ca_type = cell_array<Alloc>;
  using index_array_type = typename ca_type::index_array_type;

  auto index_array = index_array_type(num_cell_indexes - num_cells, a);

  std::size_t index = 0;
  for (std::size_t i = 0; i < num_cells; ++i) {
    std::size_t nvertex = 0;
    input >> nvertex;

    for (std::size_t j = 0; j < nvertex; ++j) {
      input >> index_array[index];
      ++index;
    }
  }

  string_type cell_types_string(a);
  input >> cell_types_string;

  if (cell_types_string != "CELL_TYPES") {
    return tl::make_unexpected(error::expected_cell_types);
  }

  std::size_t num_cell_types = 0;
  input >> num_cell_types;

  if (num_cell_types != num_cells) {
    return tl::make_unexpected(error::cell_type_count_mismatch);
  }

  using cell_type_array_type = typename ca_type::cell_type_array_type;
  auto cell_type_array = cell_type_array_type(num_cell_types, a);

  for (std::size_t i = 0; i < num_cell_types; ++i) {
    std::size_t type = 0;
    input >> type;
    cell_type_array[i] = jnum::traits<cell_type>::from_underlying(type);
  }

  return ca_type::create(index_array, cell_type_array, a);
}

template <typename Alloc>
auto parse_scalar_header(std::ifstream& input, Alloc a) -> tl::expected<scalar_header<Alloc>, error>
{
  using char_allocator_type = typename std::allocator_traits<Alloc>::template rebind_alloc<char>;
  using string_type = std::basic_string<char, std::char_traits<char>, char_allocator_type>;
  string_type scalar_string(a);
  input >> scalar_string;

  if (scalar_string != "SCALARS") {
    std::cout << "scalar_string: " << scalar_string << std::endl;
    return tl::make_unexpected(error::expected_scalar);
  }

  string_type name(a);
  input >> name;

  string_type type_string(a);
  input >> type_string;

  auto type = data_type::integer32;
  if (type_string == "int") {
    type = data_type::integer32;
  }
  else if (type_string == "long") {
    type = data_type::integer64;
  }
  else if (type_string == "float") {
    type = data_type::binary32;
  }
  else if (type_string == "double") {
    type = data_type::binary64;
  }
  else {
    return tl::make_unexpected(error::unknown_data_type);
  }

  std::size_t ncomp = 0;
  input >> ncomp;

  string_type lookup_table(a);
  input >> lookup_table;

  if (lookup_table != "LOOKUP_TABLE") {
    return tl::make_unexpected(error::expected_lookup_table);
  }

  string_type table_name(a);
  input >> table_name;

  auto header = scalar_header<Alloc>(a);
  header.set_data(type);
  header.set_name(name);
  header.set_ncomponent(ncomp);
  return header;
}

template <typename Alloc>
auto parse_scalar_data(std::ifstream& input, std::size_t num_points, Alloc a)
  -> tl::expected<nola::owning_ptr<pmdata, Alloc>, error>
{
  auto header_result = parse_scalar_header(input, a);
  if (!header_result) {
    return tl::make_unexpected(header_result.error());
  }
  auto const header = std::move(*header_result);

  auto const type = header.data();

  switch (type) {
  case data_type::integer32: {
    auto data_result = parse_scalar_data_impl<int32_t>(input, num_points, header, a);
    if (!data_result) {
      return tl::make_unexpected(data_result.error());
    }
    using data_type = scalar_data<int32_t, Alloc>;

    auto data = std::move(*data_result);
    auto sdk = nola::owning_ptr<pmdata, Alloc>(
      std::move(nola::allocate_owning<data_type, Alloc>(a, std::move(data))));
    return sdk;
  }
  case data_type::integer64: {
    auto data_result = parse_scalar_data_impl<int64_t>(input, num_points, header, a);
    if (!data_result) {
      return tl::make_unexpected(data_result.error());
    }

    using data_type = scalar_data<int64_t, Alloc>;
    auto data = std::move(*data_result);
    auto sdk = nola::owning_ptr<pmdata, Alloc>(
      std::move(nola::allocate_owning<data_type, Alloc>(a, std::move(data))));
    return sdk;
  }
  case data_type::binary32: {
    auto data_result = parse_scalar_data_impl<float>(input, num_points, header, a);
    if (!data_result) {
      return tl::make_unexpected(data_result.error());
    }

    using data_type = scalar_data<float, Alloc>;
    auto data = std::move(*data_result);
    auto sdk = nola::owning_ptr<pmdata, Alloc>(
      std::move(nola::allocate_owning<data_type, Alloc>(a, std::move(data))));
    return sdk;
  }
  case data_type::binary64: {
    auto data_result = parse_scalar_data_impl<double>(input, num_points, header, a);
    if (!data_result) {
      return tl::make_unexpected(data_result.error());
    }

    using data_type = scalar_data<double, Alloc>;
    auto data = std::move(*data_result);
    auto sdk = nola::owning_ptr<pmdata, Alloc>(
      std::move(nola::allocate_owning<data_type, Alloc>(a, std::move(data))));
    return sdk;
  }
  }
}

template <typename F, typename Alloc>
auto parse_vector_data(std::ifstream& input, Alloc a) -> tl::expected<vector_data<F, Alloc>, error>
{
  return tl::make_unexpected(error::unknown_data_type);
}

template <typename F, typename Alloc>
auto parse_scalar_data_impl(
  std::ifstream& input, std::size_t num_points, scalar_header<Alloc> sh, Alloc a)
  -> tl::expected<scalar_data<F, Alloc>, error>
{
  auto sd = scalar_data<F, Alloc>(sh.name(), num_points, sh.ncomponent(), a);
  for (std::size_t i = 0; i < num_points; ++i) {
    for (std::size_t j = 0; j < sh.ncomponent(); ++j) {
      F value = 0;
      input >> value;
      sd[i][j] = value;
    }
  }
  return sd;
}

template <typename F, typename Alloc>
auto parse_vector_data_impl(std::ifstream& input, Alloc a)
  -> tl::expected<vector_data<F, Alloc>, error>
{
}

} // namespace zell::io::vtk::ugrid::detail
#endif // ZELL_IO_VTK_UGRID_DETAIL_IO_HPP
