// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_DETAIL_METADATA_HPP
#define ZELL_IO_VTK_UGRID_DETAIL_METADATA_HPP

#include <memory>
#include <tl/expected.hpp>

namespace zell::io::vtk::ugrid::detail {
template <typename Alloc = std::allocator<void>>
class metadata {
public:
  using allocator_type = Alloc;

public:
  using title_type = std::basic_string<char, std::char_traits<char>, allocator_type>;

public:
  metadata() = default;

  explicit metadata(allocator_type const& a);

  template <typename StringType>
  metadata(
    StringType const& s,
    std::size_t ncell_scalar_data,
    std::size_t npoint_scalar_data,
    std::size_t ncell_vector_data,
    std::size_t npoint_vector_data,
    allocator_type const& a);

  template <typename StringType>
  metadata(
    StringType const& s,
    std::size_t ncell_scalar_data,
    std::size_t npoint_scalar_data,
    std::size_t ncell_vector_data,
    std::size_t npoint_vector_data);

  metadata(const metadata&) = default;
  metadata(metadata&&) noexcept = default;
  metadata(metadata const& m, allocator_type const& a);
  metadata(metadata&& m, allocator_type const& a);

  auto operator=(metadata&& m) noexcept -> metadata&;
  auto operator=(metadata const& m) noexcept -> metadata&;

  ~metadata() = default;

public:
  auto get_allocator() const noexcept -> allocator_type;

public:
  auto title() const noexcept -> title_type const&;

  auto ncell_scalar_data() const noexcept -> std::size_t;
  auto npoint_scalar_data() const noexcept -> std::size_t;

  auto ncell_vector_data() const noexcept -> std::size_t;
  auto npoint_vector_data() const noexcept -> std::size_t;

private:
  allocator_type allocator_;

  title_type title_;
  std::size_t ncell_scalar_data_{0};
  std::size_t npoint_scalar_data_{0};
  std::size_t ncell_vector_data_{0};
  std::size_t npoint_vector_data_{0};
};

// -------------------------------------------------------------------------------------------------
// Implementation
// -------------------------------------------------------------------------------------------------

template <typename Alloc>
metadata<Alloc>::metadata(allocator_type const& a) : allocator_(a)
{
}

template <typename Alloc>
metadata<Alloc>::metadata(metadata const& m, allocator_type const& a)
: allocator_(a),
  title_(m.title_, a),
  ncell_scalar_data_(ncell_scalar_data),
  npoint_scalar_data_(npoint_scalar_data),
  ncell_vector_data_(ncell_vector_data),
  npoint_vector_data_(npoint_vector_data)
{
}

template <typename Alloc>
metadata<Alloc>::metadata(metadata&& m, allocator_type const& a)
: allocator_(a),
  title_(std::move(m.title_), a),
  ncell_scalar_data_(ncell_scalar_data),
  npoint_scalar_data_(npoint_scalar_data),
  ncell_vector_data_(ncell_vector_data),
  npoint_vector_data_(npoint_vector_data)

{
}

template <typename Alloc>
template <typename StringType>
metadata<Alloc>::metadata(
  StringType const& s,
  std::size_t ncell_scalar_data,
  std::size_t npoint_scalar_data,
  std::size_t ncell_vector_data,
  std::size_t npoint_vector_data)
: metadata(s, ncell_scalar_data, npoint_scalar_data, ncell_vector_data, npoint_vector_data, Alloc())
{
}

template <typename Alloc>
template <typename StringType>
metadata<Alloc>::metadata(
  StringType const& s,
  std::size_t ncell_scalar_data,
  std::size_t npoint_scalar_data,
  std::size_t ncell_vector_data,
  std::size_t npoint_vector_data,
  allocator_type const& a)
: allocator_(a),
  title_(s, a),
  ncell_scalar_data_(ncell_scalar_data),
  npoint_scalar_data_(npoint_scalar_data),
  ncell_vector_data_(ncell_vector_data),
  npoint_vector_data_(npoint_vector_data)
{
}

template <typename Alloc>
auto metadata<Alloc>::get_allocator() const noexcept -> allocator_type
{
  return allocator_;
}

template <typename Alloc>
auto metadata<Alloc>::title() const noexcept -> title_type const&
{
  return title_;
}

template <typename Alloc>
auto metadata<Alloc>::ncell_scalar_data() const noexcept -> std::size_t
{
  return ncell_scalar_data_;
}

template <typename Alloc>
auto metadata<Alloc>::npoint_scalar_data() const noexcept -> std::size_t
{
  return npoint_scalar_data_;
}

template <typename Alloc>
auto metadata<Alloc>::ncell_vector_data() const noexcept -> std::size_t
{
  return ncell_vector_data_;
}

template <typename Alloc>
auto metadata<Alloc>::npoint_vector_data() const noexcept -> std::size_t
{
  return npoint_vector_data_;
}

} // namespace zell::io::vtk::ugrid::detail

#endif // ZELL_IO_VTK_UGRID_DETAIL_METADATA_HPP
