// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_GRID_HPP
#define ZELL_IO_VTK_UGRID_GRID_HPP

#include <memory>
#include <string>
#include <zell/io/vtk/ugrid/cell_array.hpp>
#include <zell/io/vtk/ugrid/dataset.hpp>
#include <zell/io/vtk/ugrid/point_array.hpp>

namespace zell::io::vtk::ugrid {
template <typename F, typename Alloc = std::allocator<std::byte>>
class grid {

public:
  using field_type = F;
  using allocator_type = Alloc;

private:
  using char_allocator =
    typename std::allocator_traits<allocator_type>::template rebind_alloc<char>;

public:
  using title_type = std::basic_string<char, std::char_traits<char>, char_allocator>;
  using point_array_type = point_array<field_type, allocator_type>;
  using cell_array_type = cell_array<allocator_type>;

public:
  grid() = default;
  explicit grid(allocator_type const& a);

  grid(grid&& g) noexcept = default;
  grid(grid&& g, allocator_type a) noexcept;

  auto operator=(grid&& g) noexcept -> grid& = default;

  ~grid() = default;

private:
  grid(grid const& g) = default;
  grid(grid const& g, allocator_type a);

  auto operator=(grid const& g) -> grid& = default;

public:
  auto clone() const noexcept -> grid;

  auto get_allocator() const noexcept -> allocator_type;

public:
  auto title() const noexcept -> title_type const&;

  template <typename StringType>
  void set_title(StringType const& title) noexcept;

  auto points() const noexcept -> point_array<field_type, allocator_type> const&;

  template <typename PointArrayType>
  void set_points(PointArrayType pa) noexcept;

  auto cells() const noexcept -> cell_array_type const&;

  template <typename CellArrayType>
  void set_cells(CellArrayType ca) noexcept;

private:
  allocator_type allocator_{};
  std::string title_;

  point_array_type point_array_;
  cell_array_type cell_array_;

  dataset_type point_dataset_;
  dataset_type cell_dataset_;
};

// -------------------------------------------------------------------------------------------------
// Implementation
// -------------------------------------------------------------------------------------------------

template <typename F, typename Alloc>
grid<F, Alloc>::grid(allocator_type const& a) : allocator_(a)
{
}

template <typename F, typename Alloc>
grid<F, Alloc>::grid(grid&& g, allocator_type a) noexcept
: allocator_(a), title_(std::move(g.title_), a), point_array_(std::move(g.point_array_), a)
{
}

template <typename F, typename Alloc>
auto grid<F, Alloc>::clone() const noexcept -> grid
{
  return *this;
}

template <typename F, typename Alloc>
auto grid<F, Alloc>::get_allocator() const noexcept -> allocator_type
{
  return allocator_;
}

template <typename F, typename Alloc>
auto grid<F, Alloc>::title() const noexcept -> title_type const&
{
  return title_;
}

template <typename F, typename Alloc>
template <typename StringType>
void grid<F, Alloc>::set_title(StringType const& title) noexcept
{
  title_ = title_type(title, allocator_);
}

template <typename F, typename Alloc>
auto grid<F, Alloc>::points() const noexcept -> point_array<field_type, allocator_type> const&
{
  return point_array_;
}

template <typename F, typename Alloc>
template <typename PointArrayType>
void grid<F, Alloc>::set_points(PointArrayType pa) noexcept
{
  point_array_ = point_array_type(std::move(pa), allocator_);
}

template <typename F, typename Alloc>
auto grid<F, Alloc>::cells() const noexcept -> cell_array<allocator_type> const&
{
  return cell_array_;
}

template <typename F, typename Alloc>
template <typename CellArrayType>
void grid<F, Alloc>::set_cells(CellArrayType ca) noexcept
{
  cell_array_ = cell_array_type(std::move(ca), allocator_);
}

} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_GRID_HPP
