// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_IO_HPP
#define ZELL_IO_VTK_UGRID_IO_HPP

#include <filesystem>
#include <fstream>
#include <memory>
#include <string>
#include <string_view>
#include <tl/expected.hpp>
#include <zell/io/vtk/ugrid/detail/io.hpp>
#include <zell/io/vtk/ugrid/detail/metadata.hpp>
#include <zell/io/vtk/ugrid/error.hpp>
#include <zell/io/vtk/ugrid/grid.hpp>

namespace zell::io::vtk::ugrid {
template <typename F, typename Alloc, typename StringRange>
auto read(StringRange const& filepath, Alloc a) -> tl::expected<grid<F, Alloc>, error>;

template <typename F, typename StringRange>
auto read(StringRange const& filepath) -> tl::expected<grid<F, std::allocator<std::byte>>, error>;

// -------------------------------------------------------------------------------------------------
// Implementation
// -------------------------------------------------------------------------------------------------

template <typename F, typename StringRange>
auto read(StringRange const& filepath) -> tl::expected<grid<F, std::allocator<std::byte>>, error>
{
  return read<F, std::allocator<std::byte>, StringRange>(filepath, std::allocator<std::byte>());
}

template <typename F, typename Alloc, typename StringRange>
auto read(StringRange const& filepath, Alloc a) -> tl::expected<grid<F, Alloc>, error>
{
  using string_type = std::basic_string<char, std::char_traits<char>, Alloc>;
  grid<F, Alloc> g(a);

  if (not std::filesystem::exists(filepath)) {
    return tl::make_unexpected(error::file_not_found);
  }

  std::ifstream input(filepath);

  auto metadata_result = detail::parse_header(input, a);
  if (not metadata_result.has_value()) {
    return tl::make_unexpected(metadata_result.error());
  }
  auto const metadata = std::move(*metadata_result);
  g.set_title(metadata.title());

  string_type dataset_string(a);
  input >> dataset_string;
  if (dataset_string != "DATASET") {
    return tl::make_unexpected(error::unsupported_dataset);
  }

  string_type ugrid_string(a);
  input >> ugrid_string;

  if (ugrid_string != "UNSTRUCTURED_GRID") {
    return tl::make_unexpected(error::unsupported_dataset);
  }

  auto pa_result = detail::parse_points<F>(input, a);
  if (pa_result.has_value()) {
    g.set_points(std::move(*pa_result));
  }
  else {
    return tl::make_unexpected(pa_result.error());
  }

  auto ca_result = detail::parse_cells(input, a);

  if (ca_result.has_value()) {
    g.set_cells(std::move(*ca_result));
  }
  else {
    return tl::make_unexpected(ca_result.error());
  }

  auto const point_data_count = metadata.npoint_scalar_data() + metadata.npoint_vector_data();

  if (point_data_count != 0) {
    string_type point_dataset_string(a);
    input >> point_dataset_string;

    if (point_dataset_string != "POINT_DATA") {
      return tl::make_unexpected(error::expected_point_data);
    }

    std::size_t num_points = 0;
    input >> num_points;
    if (num_points != g.points().size()) {
      return tl::make_unexpected(error::point_data_size_mismatch);
    }

    auto pd = dataset<Alloc>(point_data_count, a);

    for (std::size_t i = 0; i < metadata.npoint_scalar_data(); ++i) {
      auto sd_result = detail::parse_scalar_data(input, num_points, a);
      if (sd_result.has_value()) {
        pd.add_data(std::move(*sd_result));
      }
      else {
        return tl::make_unexpected(sd_result.error());
      }
    }
  }

  // if (metadata.ncell_scalar_data() != 0 or metadata.ncell_vector_data() != 0) {
  //   auto cd_result = detail::parse_cell_dataset<Alloc>(input, a, metadata);
  //   if (cd_result.has_value()) {
  //   }
  //   else {
  //     return tl::make_unexpected(cd_result.error());
  //   }
  // }

  return g;
}

} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_IO_HPP
