// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_POINT_ARRAY_HPP
#define ZELL_IO_VTK_UGRID_POINT_ARRAY_HPP

#include <bati/vector.hpp>
#include <memory>
#include <vector>

namespace zell::io::vtk::ugrid {
template <typename F, typename Alloc = std::allocator<std::byte>>
class point_array {
public:
  using field_type = F;
  using value_type = bati::vector<field_type, 3>;
  using reference = value_type&;
  using const_reference = value_type const&;
  using allocator_type = Alloc;

private:
  using value_allocator =
    typename std::allocator_traits<allocator_type>::template rebind_alloc<value_type>;

  using storage_type = std::vector<value_type, value_allocator>;

public:
  using pointer = typename storage_type::pointer;
  using const_pointer = typename storage_type::const_pointer;

  using iterator = typename storage_type::iterator;
  using const_iterator = typename storage_type::const_iterator;

  using size_type = std::size_t;
  using difference_type = std::ptrdiff_t;

  using reverse_iterator = std::reverse_iterator<iterator>;
  using const_reverse_iterator = std::reverse_iterator<const_iterator>;

public:
  point_array() = default;

  explicit point_array(size_type size);
  explicit point_array(allocator_type const& a);
  explicit point_array(size_type size, allocator_type const& a);

  point_array(point_array&& g) noexcept = default;
  point_array(point_array&& g, allocator_type a) noexcept;

  auto operator=(point_array&& g) noexcept -> point_array& = default;

  ~point_array() = default;

private:
  point_array(point_array const& g) = default;
  point_array(point_array const& g, allocator_type a);

  auto operator=(point_array const& g) -> point_array& = default;

public:
  auto clone() const -> point_array;

public:
  auto size() const noexcept -> size_type;

  auto begin() const noexcept -> const_iterator;
  auto begin() noexcept -> iterator;

  auto end() const noexcept -> const_iterator;
  auto end() noexcept -> iterator;

  auto operator[](size_type index) noexcept -> reference;
  auto operator[](size_type index) const noexcept -> const_reference;

private:
  storage_type array_;
};

template <typename F, typename Alloc>
point_array<F, Alloc>::point_array(size_type size) : array_(size)
{
}
template <typename F, typename Alloc>
point_array<F, Alloc>::point_array(size_type size, allocator_type const& a) : array_(size, a)
{
}

template <typename F, typename Alloc>
point_array<F, Alloc>::point_array(allocator_type const& a) : array_(a)
{
}

template <typename F, typename Alloc>
point_array<F, Alloc>::point_array(point_array&& g, allocator_type a) noexcept
: array_(std::move(g.array_), a)
{
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::clone() const -> point_array
{
  return point_array(*this);
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::size() const noexcept -> size_type
{
  return array_.size();
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::begin() const noexcept -> const_iterator
{
  return array_.begin();
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::begin() noexcept -> iterator
{
  return array_.begin();
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::end() const noexcept -> const_iterator
{
  return array_.end();
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::end() noexcept -> iterator
{
  return array_.end();
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::operator[](size_type index) noexcept -> reference
{
  return array_[index];
}

template <typename F, typename Alloc>
auto point_array<F, Alloc>::operator[](size_type index) const noexcept -> const_reference
{
  return array_[index];
}
} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_POINT_ARRAY_HPP
