// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_SCALAR_DATA_HPP
#define ZELL_IO_VTK_UGRID_SCALAR_DATA_HPP

#include <string>
#include <zell/io/vtk/ugrid/detail/scalar_data_base.hpp>
#include <zell/io/vtk/ugrid/scalar_value.hpp>
#include <zell/io/vtk/ugrid/type_map.hpp>

namespace zell::io::vtk::ugrid {

template <typename F, typename Alloc = std::allocator<F>>
class scalar_data : public detail::scalar_data_base {
public:
  using field_type = F;
  using value_type = scalar_value<field_type>;
  using allocator_type = Alloc;

private:
  using alloc_traits = std::allocator_traits<allocator_type>;
  using char_allocator = typename alloc_traits::template rebind_alloc<char>;

  using value_allocator = typename alloc_traits::template rebind_alloc<value_type>;
  using value_container = std::vector<value_type, value_allocator>;

  using field_allocator = typename alloc_traits::template rebind_alloc<field_type>;
  using field_container = std::vector<field_type, field_allocator>;

public:
  using string_type = std::basic_string<char, std::char_traits<char>, char_allocator>;

  using reference = typename value_container::reference;
  using const_reference = typename value_container::const_reference;

  using iterator = typename value_container::iterator;
  using const_iterator = typename value_container::const_iterator;

  using reverse_iterator = typename value_container::reverse_iterator;
  using const_reverse_iterator = typename value_container::const_reverse_iterator;

  using size_type = typename value_container::size_type;
  using difference_type = typename value_container::difference_type;

public:
  scalar_data() = default;
  scalar_data(scalar_data&&) noexcept = default;
  auto operator=(scalar_data&&) noexcept -> scalar_data& = default;
  ~scalar_data() override = default;

  explicit scalar_data(allocator_type const& alloc);
  explicit scalar_data(
    string_type const& name,
    std::size_t nelement,
    std::size_t ncomponent,
    allocator_type const& alloc);

private:
  auto operator=(scalar_data const&) -> scalar_data& = default;
  scalar_data(scalar_data const&) = default;

  template <typename Allocator>
  scalar_data(scalar_data const&, Allocator const& alloc);

public:
  auto clone() const -> scalar_data;

  template <typename Allocator>
  auto clone(Allocator const& alloc) const -> scalar_data<F, Allocator>;

public:
  auto name() const noexcept -> std::string const& final;
  auto ncomponent() const noexcept -> std::size_t final;
  auto data() const noexcept -> struct data_type final;

public:
  constexpr auto begin() const noexcept -> const_iterator;
  constexpr auto end() const noexcept -> const_iterator;

  constexpr auto begin() noexcept -> iterator;
  constexpr auto end() noexcept -> iterator;

  constexpr auto operator[](size_type index) const noexcept -> const_reference;
  constexpr auto operator[](size_type index) noexcept -> reference;

  constexpr auto size() noexcept -> std::size_t;

private:
  allocator_type allocator_;
  string_type name_;
  std::size_t ncomponent_{};
  value_container value_array_;
  field_container field_array_;
};

template <typename F, typename Alloc>
scalar_data<F, Alloc>::scalar_data(allocator_type const& alloc)
: allocator_{alloc}, name_{alloc}, value_array_{alloc}, field_array_{alloc}
{
}

template <typename F, typename Alloc>
scalar_data<F, Alloc>::scalar_data(
  string_type const& name,
  std::size_t nelement,
  std::size_t ncomponent,
  allocator_type const& alloc)
: allocator_{alloc},
  name_{name},
  ncomponent_{ncomponent},
  value_array_(nelement, alloc),
  field_array_(nelement * ncomponent, alloc)
{
  for (std::size_t i = 0; i < nelement; ++i) {
    value_array_[i] = value_type(ncomponent, &field_array_[i * ncomponent]);
  }
}

template <typename F, typename Alloc>
auto scalar_data<F, Alloc>::clone() const -> scalar_data
{
  return *this;
}

template <typename F, typename Alloc>
template <typename Allocator>
auto scalar_data<F, Alloc>::clone(Allocator const& alloc) const -> scalar_data<F, Allocator>
{
  return scalar_data(*this, alloc);
}

template <typename F, typename Alloc>
auto scalar_data<F, Alloc>::name() const noexcept -> std::string const&
{
  return name_;
}

template <typename F, typename Alloc>
auto scalar_data<F, Alloc>::ncomponent() const noexcept -> std::size_t
{
  return ncomponent_;
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::begin() const noexcept -> const_iterator
{
  return value_array_.begin();
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::end() const noexcept -> const_iterator
{
  return value_array_.end();
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::begin() noexcept -> iterator
{
  return value_array_.begin();
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::end() noexcept -> iterator
{
  return value_array_.end();
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::operator[](size_type index) const noexcept -> const_reference
{
  return value_array_[index];
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::operator[](size_type index) noexcept -> reference
{
  return value_array_[index];
}

template <typename F, typename Alloc>
constexpr auto scalar_data<F, Alloc>::size() noexcept -> std::size_t
{
  return value_array_.size();
}

template <typename F, typename Alloc>
auto scalar_data<F, Alloc>::data() const noexcept -> data_type
{
  return reverse_type_map<F>::value;
}

} // namespace zell::io::vtk::ugrid
#endif // ZELL_IO_VTK_UGRID_SCALAR_DATA_HPP
