// -------------------------------------------------------------------------------------------------
// SPDX-License-Identifier: Apache-2.0
// SPDX-FileCopyrightText: (C) 2022 Jayesh Badwaik <j.badwaik@fz-juelich.de>
// -------------------------------------------------------------------------------------------------

#ifndef ZELL_IO_VTK_UGRID_SCALAR_VALUE_HPP
#define ZELL_IO_VTK_UGRID_SCALAR_VALUE_HPP

#include <nola/wrap_iterator.hpp>
#include <vector>

namespace zell::io::vtk::ugrid {

template <typename F>
class scalar_value {

public:
  using field_type = F;
  using value_type = field_type;

  using reference = value_type&;
  using const_reference = value_type const&;

  using pointer = value_type*;
  using const_pointer = value_type const*;

  using iterator = nola::wrap_iterator<pointer>;
  using const_iterator = nola::wrap_iterator<const_pointer>;

  using size_type = std::size_t;
  using difference_type = std::ptrdiff_t;

  using reverse_iterator = std::reverse_iterator<iterator>;
  using const_reverse_iterator = std::reverse_iterator<const_iterator>;

public:
  scalar_value() = default;
  scalar_value(std::size_t ncomp, field_type* begin);

public:
  constexpr auto begin() const noexcept -> const_iterator { return const_iterator{begin_}; }
  constexpr auto end() const noexcept -> const_iterator { return const_iterator{end_}; }

  constexpr auto begin() noexcept -> iterator { return iterator{begin_}; }
  constexpr auto end() noexcept -> iterator { return iterator{end_}; }

  constexpr auto operator[](size_type index) const noexcept -> const_reference
  {
    return begin_[index];
  }
  constexpr auto operator[](size_type index) noexcept -> reference { return begin_[index]; }

  constexpr auto size() noexcept -> std::size_t { return end_ - begin_; }

private:
  field_type* begin_;
  field_type* end_;
};

template <typename F>
scalar_value<F>::scalar_value(std::size_t ncomp, field_type* begin)
: begin_{begin}, end_{begin + ncomp}
{
}
} // namespace zell::io::vtk::ugrid

#endif // ZELL_IO_VTK_UGRID_SCALAR_VALUE_HPP
